/*
*************************************************************
* File: LCD_API.c
* Description: This file contains the driver routines in ANSI-C, 
*              of Character LCD Driver for Z8 Encore! MCU
*
* Copyright 2004 ZiLOG Inc. ALL RIGHTS RESERVED.
*
* The source code in this file was written by an
* authorized ZiLOG employee or a licensed consultant.
* The source code has been verified to the fullest
* extent possible.
*
* Permission to use this code is granted on a royalty-free
* basis. However users are cautioned to authenticate the
* code contained herein.
*
* ZiLOG DOES NOT GUARANTEE THE VERACITY OF THE SOFTWARE.
************************************************************* */
/*
******************************************************************************
**  Includes
******************************************************************************
*/
#include "LCD_API.h"	/* Header file */

/*
******************************************************************************
**  Global Declarations
******************************************************************************
*/
unsigned int volatile current_ms;	// Used by delayms() & isr_timer0()

/*
******************************************************************************
**
**  Routine:
**      LCD_init
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function initializes the LCD as per parameters in LCD_API.h
**
******************************************************************************
*/
void LCD_init (void)
{
	PEOUT = 0xF0;
	delay();

	PEOUT = 0x34;			// Function Set (00110100)
	PEOUT = 0x30;			// 00110000	- Enable Low
	delay();				// Provide some delay

	PEOUT = 0x34;			// Repeat
	PEOUT = 0x30;
	delay();

	PEOUT = 0x34;			// Repeat
	PEOUT = 0x30;
	delay();

	PEOUT = 0x24;			// 00100100 - Enable High
	PEOUT = 0x20;			// 00100000 - Enable Low
	delay();

	LCD_command(0x2C);	// Function set (4-bit, 2-lines, 5x10 dots)
	LCD_off();
	LCD_clear();
	LCD_on();
	LCD_home();
	LCD_blink();
}	/* LCD_init */

/*
******************************************************************************
**
**  Routine:
**      LCD_on
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function puts On display of the LCD
**
******************************************************************************
*/
void LCD_on (void)			// Display ON + Cursor ON + Blink OFF = 00001110
{
	while (LCD_checkbusy()) ;
	LCD_command(0x0C);
}	/* LCD_on */

/*
******************************************************************************
**
**  Routine:
**      LCD_off
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function puts Off display of the LCD
**
******************************************************************************
*/
void LCD_off (void)			// Display Off + Cursor/Blink Off = 00001000
{
	while (LCD_checkbusy()) ;
	LCD_command(0x08);
}	/* LCD_off */

/*
******************************************************************************
**
**  Routine:
**      LCD_command
**
**  Parameters:
**      Instruction word to be written
**
**  Return:
**      void
**
**  Purpose:
**      This function sends Instruction to LCD controller
**
******************************************************************************
*/
void LCD_command (unsigned char command_word)
{
	unsigned char data_value;	// Temporary variable
	while (LCD_checkbusy()) ;

	data_value = command_word & 0xF0;
	data_value |= 0x04;
	PEOUT = data_value;		// Send instruction High nibble to LCD
	data_value &= 0xFB;
	PEOUT = data_value;		// Put Enable to Low

	data_value = command_word << 4;
	data_value |= 0x04;
	PEOUT = data_value;		// Send instruction Low nibble to LCD
	data_value &= 0xFB;
	PEOUT = data_value;		// Put Enable to Low
}	/* LCD_command */

/*
******************************************************************************
**
**  Routine:
**      LCD_clear
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function clears LCD screen & sets DDRAM to 0
**
******************************************************************************
*/
void LCD_clear (void)
{
	while (LCD_checkbusy()) ;
	LCD_command(0x01);
}	/* LCD_clear */

/*
******************************************************************************
**
**  Routine:
**      LCD_blink
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function puts cursor in blink mode
**
******************************************************************************
*/
void LCD_blink (void)
{
	while (LCD_checkbusy()) ;
	LCD_command(0x0D);
}	/* LCD_blink */

/*
******************************************************************************
**
**  Routine:
**      LCD_blink_off
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function puts off cursor blink mode
**
******************************************************************************
*/
void LCD_blink_off (void)
{
	while (LCD_checkbusy()) ;
	LCD_command(0x0C);
}	/* LCD_blink_off */

/*
******************************************************************************
**
**  Routine:
**      LCD_home
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function brings cursor to home position
**
******************************************************************************
*/
void LCD_home (void)
{
	while (LCD_checkbusy()) ;
	LCD_command(0x02);
}	/* LCD_home */

/*
******************************************************************************
**
**  Routine:
**      LCD_curpos
**
**  Parameters:
**      void
**
**  Return:
**      Current cursor position in DDRAM
**
**  Purpose:
**      This function reads & returns the current cursor position in DDRAM
**
******************************************************************************
*/
unsigned char LCD_curpos (void)
{
	unsigned char add_readH = 0, add_readL = 0;

	while (LCD_checkbusy()) ;
	PEADDR = DATA_DIR;		// Data direction control for LCD PortE
	PECTL = 0xF0;			// D7-4 configured as inputs
	PEOUT = 0x06;			// Enable address read pulse
	PEOUT = 0x02;			// Enable Low
	add_readH = PEIN;		// Read port data
	PEOUT = 0x06;			// Enable address read pulse
	PEOUT = 0x02;			// Enable Low
	add_readL = PEIN;		// Read port data
	add_readH &= 0x70;		// Strip off unused bits
	add_readL &= 0xF0;		// Strip off unused bits
	PEADDR = DATA_DIR;		// Data direction control for LCD PortE
	PECTL = 0x00;			// D7-4 reconfigured as outputs
	PEOUT = 0x00;			// Outputs made Low
	add_readL >>= 4;		// Take only Lower nibble
	add_readH |= add_readL;	// Merge data nibbles
	return add_readH ;
}	/* LCD_curpos */

/*
******************************************************************************
**
**  Routine:
**      LCD_writedata
**
**  Parameters:
**      Data word to be written
**
**  Return:
**      void
**
**  Purpose:
**      This function writes a data word to the LCD controller
**
******************************************************************************
*/
void LCD_writedata (unsigned char data_word)
{
	unsigned char data_value;	// Temporary variable
	while (LCD_checkbusy()) ;

	data_value = data_word & 0xF0;
	data_value |= 0x0C;
	PEOUT = data_value;		// Send instruction High nibble to LCD
	data_value &= 0xFB;
	PEOUT = data_value;		// Put Enable to Low

	data_value = data_word << 4;
	data_value |= 0x0C;
	PEOUT = data_value;		// Send instruction Low nibble to LCD
	data_value &= 0xFB;
	PEOUT = data_value;		// Put Enable to Low
}	/* LCD_writedata */

/*
******************************************************************************
**
**  Routine:
**      LCD_setposition
**
**  Parameters:
**      Takes row number (0/1/2/3) & column number (0 - 39) as input
**
**  Return:
**      void
**
**  Purpose:
**      This function sets cursor at indicated position for printing
**
******************************************************************************
*/
void LCD_setposition (unsigned char row_number, unsigned char column_number)
{
	unsigned char calc_position = 0;
	while (LCD_checkbusy()) ;
	switch (row_number)
	{
		case 0 :
			calc_position = start_address_0 ;
			break;
		default :
			calc_position = start_address_1 ;
	}
	calc_position += column_number ;
	LCD_set_ddram(calc_position);
}	/* LCD_setposition */

/*
******************************************************************************
**
**  Routine:
**      LCD_printc
**
**  Parameters:
**      Takes ASCII symbol value as input
**
**  Return:
**      void
**
**  Purpose:
**      This function prints a character at cursor position
**
******************************************************************************
*/
void LCD_printc (unsigned char character)
{
	while (LCD_checkbusy()) ;
	LCD_writedata(character);
}	/* LCD_printc */

/*
******************************************************************************
**
**  Routine:
**      LCD_prints
**
**  Parameters:
**      Takes string or address as input
**
**  Return:
**      void
**
**  Purpose:
**      This function prints a string starting at cursor position
**		and also positions the characters inside the visible space
**
******************************************************************************
*/
void LCD_prints (unsigned char *string)
{
	unsigned char current_position ;
	while (LCD_checkbusy()) ;
	current_position = LCD_curpos() ;
	while (*string)
	{
		if ((current_position >= end_address_0)
					&& (current_position < start_address_1))
		{
			LCD_setposition(1,0) ;
			current_position = start_address_1 ;
		}
		LCD_printc(*string++) ;
		current_position++ ;
	}
}	/* LCD_prints */

/*
******************************************************************************
**
**  Routine:
**      LCD_scroll
**
**  Parameters:
**		Pointer to text location
**
**  Return:
**      void
**
**  Purpose:
**      This function scrolls a string in several screens
**
******************************************************************************
*/
void LCD_scroll (unsigned char *long_text)
{
	int count ;
	while (1)
	{
		count = 0 ;
		LCD_clear();
		while ((count < columns) && (*long_text != NULL))
		{
			LCD_printc(*long_text++);
			count++ ;
		}
		LCD_setposition(1,0);
		count = 0 ;
		while ((count < columns) && (*long_text != NULL))
		{
			LCD_printc(*long_text++);
			count++ ;
		}
		if (*long_text == NULL)
			break ;
		count = (columns * (rows - 1)) ;
		while (count--)
			long_text-- ;	// Go back by a line
		delayms(1000);		// Provide 1 sec delay between screens
	}
	delayms(2500);		// Provide 2.5 sec delay at the end of show
}	/* LCD_scroll */

/*
******************************************************************************
**
**  Routine:
**      LCD_readchar
**
**  Parameters:
**      void
**
**  Return:
**      Character read from current location
**
**  Purpose:
**      This function reads a character at current cursor position
**
******************************************************************************
*/
unsigned char LCD_readchar (void)
{
	unsigned char data_readH = 0, data_readL = 0;

	while (LCD_checkbusy()) ;
	PEADDR = DATA_DIR;		// Data direction control for LCD PortE
	PECTL = 0xF0;			// D7-4 configured as inputs
	PEOUT = 0x0E;			// Enable data read pulse
	PEOUT = 0x0A;			// Enable Low
	data_readH = PEIN;
	data_readH &= 0xF0;		// Strip off unused bits
	PEOUT = 0x0E;			// Enable address read pulse
	PEOUT = 0x0A;			// Enable Low
	data_readL = PEIN;
	data_readL &= 0xF0;		// Strip off unused bits
	PEADDR = DATA_DIR;		// Data direction control for LCD PortE
	PECTL = 0x00;			// D7-4 reconfigured as outputs
	PEOUT = 0x00;
	data_readL >>= 4;
	data_readH |= data_readL;
	return data_readH ;
}	/* LCD_readchar */

/*
******************************************************************************
**
**  Routine:
**      LCD_set_cgram
**
**  Parameters:
**		CGRAM address to be set
**
**  Return:
**		void
**
**  Purpose:
**      This function sets CGRAM address location
**
******************************************************************************
*/
void LCD_set_cgram (unsigned char cgram_address)
{
	cgram_address |= 0x40 ;		// To set DB6
	LCD_command(cgram_address);
}	/* LCD_set_cgram */

/*
******************************************************************************
**
**  Routine:
**      LCD_set_ddram
**
**  Parameters:
**		DDRAM address to be set
**
**  Return:
**		void
**
**  Purpose:
**      This function sets DDRAM address location
**
******************************************************************************
*/
void LCD_set_ddram (unsigned char ddram_address)
{
	ddram_address |= 0x80 ;		// To set DB7
	LCD_command(ddram_address);
}	/* LCD_set_ddram */

/*
******************************************************************************
**
**  Routine:
**      LCD_copymap
**
**  Parameters:
**		Pointer to user bitmap
**
**  Return:
**		void
**
**  Purpose:
**      This function loads eight user-defined character sets to CGRAM
**
******************************************************************************
*/
void LCD_copymap (unsigned char *bitmap)
{
	unsigned char counter = 64;
	LCD_set_cgram(0);
	while (counter--)
		LCD_writedata(*bitmap++);
}	/* LCD_copymap */

/*
******************************************************************************
**
**  Routine:
**		LCD_fillchar
**
**  Parameters:
**		Symbol & number of repetitions
**
**  Return:
**      void
**
**  Purpose:
**      This function fills number of memory locations with ASCII symbol
**
******************************************************************************
*/
void LCD_fillchar (unsigned char ascii_char, unsigned char repeat)
{
	while (repeat--)
		LCD_printc(ascii_char);
}	/* LCD_fillchar */

/*
******************************************************************************
**
**  Routine:
**		LCD_insert
**
**  Parameters:
**		void
**
**  Return:
**      void
**
**  Purpose:
**      This function inserts a blank space at current cursor location
**
******************************************************************************
*/
void LCD_insert (void)
{
	unsigned char insert_position, buffer[columns + 1], position_counter = 0 ;

	insert_position = LCD_curpos() ;
	LCD_set_ddram(insert_position) ;
	buffer[position_counter++] = ' ' ;		// Insert blank space
	while (position_counter < columns)
		buffer[position_counter++] = LCD_readchar();
	buffer[position_counter] = NULL ;		// Append NULL at the end
	LCD_set_ddram(insert_position) ;
	LCD_prints(buffer) ;
	LCD_set_ddram(insert_position) ;
}	/* LCD_insert */

/*
******************************************************************************
**
**  Routine:
**		LCD_backspace
**
**  Parameters:
**		void
**
**  Return:
**      void
**
**  Purpose:
**      This function deletes one character before current position
**
******************************************************************************
*/
void LCD_backspace (void)
{
	unsigned char present_position ;
	present_position = LCD_curpos() - 1 ;
	LCD_set_ddram(present_position) ;
	LCD_printc(' ') ;
	LCD_set_ddram(present_position) ;
}	/* LCD_backspace */

/*
******************************************************************************
**
**  Routine:
**      delay
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function provides fixed delay during initializing of LCD
**
******************************************************************************
*/
void delay (void)
{
	int i;
	i = 0x02ff;
	do {
	  i--;
	} while (i);
}	/* delay */

/*
******************************************************************************
**
**  Routine:
**      delayms
**
**  Parameters:
**		Number of milliseconds
**
**  Return:
**      void
**
**  Purpose:
**      This function provides delay in milliseconds (maximum 65535 ms)
**
******************************************************************************
*/
void delayms (unsigned int milliseconds)
{
	current_ms = 0;
	T0CTL &= 0x7F;				// Disable T0
    T0H = 0x00 ;				// Restart from 0
	T0L = 0x00 ;
	T0CTL |= 0x80 ;				// Put On T0 
	while (current_ms < milliseconds)
		;						// Delay loop
	T0CTL &= 0x7F;				// Stop T0
}	/* delayms */

/*
******************************************************************************
**
**  Routine:
**		init_timer0
**
**  Parameters:
**		void
**
**  Return:
**      void
**
**  Purpose:
**      This function initializes Timer0
**
******************************************************************************
*/
void init_timer0(void)
{
	SET_VECTOR(TIMER0, isr_timer0);
	T0H = 0x00;				// Start value = 0
	T0L = 0x00;
	T0CPH = 0x48;			// 1ms reload counter
	T0CPL = 0x00;
	T0CTL = 0x01;			// Continuous mode; clock divide by 1
	IRQ0E0 &= 0xDF;
	IRQ0E1 |= 0x20;			// T0 has lowest priority
}	/* init_timer0 */

/*
******************************************************************************
**
**  Routine:
**		isr_timer0
**
**  Parameters:
**		void
**
**  Return:
**      void
**
**  Purpose:
**      This ISR is called on completion of every 1ms
**
******************************************************************************
*/
#pragma interrupt
void isr_timer0(void)
{
	current_ms++ ;	// Increment count
}		/* isr_timer0 */

/*
******************************************************************************
**
**  Routine:
**      LCD_checkbusy
**
**  Parameters:
**      void
**
**  Return:
**      Busy = 1, otherwise 0
**
**  Purpose:
**      This function checks if LCD is busy, before a command
**
******************************************************************************
*/
unsigned char LCD_checkbusy (void)
{
	unsigned char check_busy = 0;

	PEADDR = DATA_DIR;		// Data direction control for LCD PortE
	PECTL = 0xF0;			// D7-4 configured as inputs
	PEOUT = 0x06;			// Enable data read pulse
	PEOUT = 0x02;			// Enable Low
	check_busy = PEIN;		// Gather data
	PEOUT = 0x06;			// Enable data read pulse second time
	PEOUT = 0x02;			// Enable Low, but data not needed
	check_busy &= 0x80;		// Strip off unused bits
	PEADDR = DATA_DIR;		// Data direction control for LCD PortE
	PECTL = 0x00;			// D7-4 reconfigured as outputs
	PEOUT = 0x00;			// Place a Low on out pins
	return check_busy;
}	/* LCD_checkbusy */

/*
******************************************************************************
**
**  Routine:
**      LCD_printc_user
**
**  Parameters:
**      User defined character number to be printed
**
**  Return:
**      void
**
**  Purpose:
**      This function displays a user defined character at current cursor position
**
******************************************************************************
*/
void LCD_printc_user (unsigned char number)
{
	unsigned char current_position, char_symbol;
	current_position = LCD_curpos();		// Read current cursor position
	LCD_set_cgram(number);					// Set character location in CGRAM
	char_symbol = LCD_readchar();			// Read user defined symbol
	LCD_set_ddram(current_position);		// Set cursor position on screen
	LCD_printc(char_symbol);				// Display user defined symbol
} /* LCD_printc_user */

/*
******************************************************************************
**
**  Routine:
**      LCD_gpio_set
**
**  Parameters:
**      void
**
**  Return:
**      void
**
**  Purpose:
**      This function configures the GPIO pins of Z8F
**
******************************************************************************
*/
void LCD_gpio_set (void)
{
	PEADDR = DATA_DIR;		// Data direction control for LCD PortE
	PECTL = 0x00;
	PEADDR = OUT_CTL;		// Setup PortE pins as output
	PECTL = 0x00;
	PEADDR = ALT_FUN;		// Disable alternate functions
	PECTL = 0x00;
	PEOUT = 0x00;			// Set PortE LCD pins to Low level
} /* LCD_gpio_set */

/*
******************************************************************************
**  End of File
******************************************************************************
*/